"""
Haptic controller (TouchX) for virtual mobile IO.
"""

import traceback
import threading
import time
import os
import numpy as np
import hebi
from scipy.spatial.transform import Rotation as R

from hebi_tools.hebi_proto import FirmwareInfo

from hebi_haptic import HapticArmControl, Floating, Idle


def moving_average_filter(buffer, new_value):
    """Apply moving average filter to a new value using the provided buffer."""
    if isinstance(new_value, np.ndarray):
        # For numpy arrays (like linear_velocity, angular_velocity)
        buffer.append(new_value.copy())
        if len(buffer) > 0:
            return np.mean(buffer, axis=0)
        else:
            return new_value
    else:
        # For scalar values (like individual joint velocities)
        buffer.append(new_value)
        if len(buffer) > 0:
            return np.mean(buffer)
        else:
            return new_value


class HebiHaptic:
    def __init__(self):
        self.shutdown = False
        self.disconnected = True
        self.device_lock = threading.Lock()
        self.haptic_device: 'HapticArmControl | None' = None
        self.hammer = False
        self.trigger = False

    def initialize_haptic_device(self):
        print("Initializing haptic device...")

        try:
            l = hebi.Lookup()
            time.sleep(0.2)  # Give haptic device time to initialize
            cfg_path = os.path.join(os.path.dirname(__file__), 'config', 'haptic-arm.cfg.yaml')
            cfg = hebi.config.load_config(cfg_path)
            self.haptic_device = HapticArmControl(l, cfg)
            self.disconnected = False
            print("Haptic device initialized successfully.")
            self.haptic_device.start()
            time.sleep(0.15)
            #self.haptic_device.request_transition(Floating(allow_tool_force=True), blocking=True)
            return True
    
    #######################

        except Exception as e:
            print(f"Haptic device initialization error: {e}\n{traceback.format_exc()}")
            self.disconnected = True
            return False

    def cleanup_haptic_device(self):
        """Clean up the haptic device"""
        # Clean up haptic device
        if self.haptic_device is not None:
            try:
                self.haptic_device.stop()
                print("Haptic device stopped.")
            except Exception as e:
                print(f"Error stopping haptic device: {e}")

        self.disconnected = True

    def run(self):
        print("Starting haptic device controller...")

        # Initialize haptic device
        if not self.initialize_haptic_device():
            print("Failed to initialize haptic device.")
            return

        try:
            while not self.shutdown:
                if self.disconnected:
                    print("Haptic device disconnected. Attempting to reconnect...")
                    time.sleep(1.0)
                    continue

                if self.haptic_device:
                    self.trigger = self.haptic_device.io_fbk.io.b.get_int(1) == 0
                    self.hammer = self.haptic_device.io_fbk.io.b.get_int(2) == 0
                    grip = self.haptic_device.io_fbk.io.b.get_int(3) == 0

                    if grip:
                        self.haptic_device.request_transition(Floating(allow_tool_force=True), blocking=True)
                    else:
                        self.haptic_device.request_transition(Idle(), blocking=True)



                # Haptic device updates are handled in the callback
                time.sleep(0.01)  # Small sleep to prevent excessive CPU usage

        except KeyboardInterrupt:
            print("\nShutting down haptic controller...")
            self.shutdown = True  # Set shutdown flag when keyboard interrupt is caught
        except Exception as e:
            print(f"Error in haptic controller: {e}")
        finally:
            self.cleanup_haptic_device()

    def device_handler(self, device, request, response):
        response.sender_id = device.sender_id
        response.echo.tx_time = request.echo.tx_time
        response.echo.payload = request.echo.payload
        response.echo.sequence_number = request.echo.sequence_number

        return_response = False

        if request.HasField("settings"):
            if request.settings.HasField("name"):
                if request.settings.name.HasField("name"):
                    device.name = request.settings.name.name
                if request.settings.name.HasField("family"):
                    device.family = request.settings.name.family

        if request.request_settings:
            response.settings.name.name = device.name
            response.settings.name.family = device.family
            return_response = True
        else:
            response.settings.ClearField("name")

        if request.request_firmware_info:
            response.firmware_info.type = device.fw_type
            response.firmware_info.revision = device.fw_rev
            response.firmware_info.mode = FirmwareInfo.APPLICATION
            return_response = True
        else:
            response.ClearField("firmware_info")

        if request.request_ethernet_info:
            response.ethernet_info.mac_address = device.mac_address
            response.ethernet_info.ip_address = device.ip_address
            response.ethernet_info.netmask = device.netmask
            return_response = True
        else:
            response.ClearField("ethernet_info")

        if request.request_hardware_info:
            response.hardware_info.serial_number = device.hw_serial_number
            response.hardware_info.mechanical_type = device.hw_mechanical_type
            response.hardware_info.mechanical_revision = device.hw_mechanical_revision
            response.hardware_info.electrical_type = device.hw_electrical_type
            response.hardware_info.electrical_revision = device.hw_electrical_revision
            return_response = True
        else:
            response.ClearField("hardware_info")

        if request.request_feedback and not self.disconnected:
            if self.haptic_device is not None:
                
                response.feedback.mobile_feedback.ar_position.x = self.haptic_device.ee_xyz[0]
                response.feedback.mobile_feedback.ar_position.y = self.haptic_device.ee_xyz[1]
                response.feedback.mobile_feedback.ar_position.z = self.haptic_device.ee_xyz[2]

                q = self.haptic_device.get_corrected_orientation().as_quat().flatten()
                response.feedback.mobile_feedback.ar_orientation.x = q[0]
                response.feedback.mobile_feedback.ar_orientation.y = q[1]
                response.feedback.mobile_feedback.ar_orientation.z = q[2]
                response.feedback.mobile_feedback.ar_orientation.w = q[3]

                # TODO: add buttons when hardware prototype done
                response.feedback.io_feedback.b.pin1.int_value = 1 if self.hammer else 0
                response.feedback.io_feedback.b.pin2.int_value = 1 if self.trigger else 0

                #response.feedback.io_feedback.a.pin1.float_value = device_state.joints[0]
                #response.feedback.io_feedback.a.pin2.float_value = device_state.joints[1]
                #response.feedback.io_feedback.a.pin3.float_value = device_state.joints[2]
                #response.feedback.io_feedback.a.pin4.float_value = device_state.joints[3]
                #response.feedback.io_feedback.a.pin5.float_value = device_state.joints[4]
                #response.feedback.io_feedback.a.pin6.float_value = device_state.joints[5]

                response.feedback.io_feedback.f.pin1.float_value = self.haptic_device.force_fbk[0]
                response.feedback.io_feedback.f.pin2.float_value = self.haptic_device.force_fbk[1]
                response.feedback.io_feedback.f.pin3.float_value = self.haptic_device.force_fbk[2]

                response.feedback.wrench.force.x = self.haptic_device.force_fbk[0]
                response.feedback.wrench.force.y = self.haptic_device.force_fbk[1]
                response.feedback.wrench.force.z = self.haptic_device.force_fbk[2]

                response.feedback.gyro.x = self.haptic_device.io_fbk[0].gyro[0]
                response.feedback.gyro.y = self.haptic_device.io_fbk[0].gyro[1]
                response.feedback.gyro.z = self.haptic_device.io_fbk[0].gyro[2]

                response.feedback.accel.x = self.haptic_device.io_fbk[0].accelerometer[0]
                response.feedback.accel.y = self.haptic_device.io_fbk[0].accelerometer[1]
                response.feedback.accel.z = self.haptic_device.io_fbk[0].accelerometer[2]

            return_response = True
        else:
            response.ClearField("feedback")

        if request.HasField("command") and not self.disconnected:
            # Read force commands from f IO pins
            force_x, force_y, force_z = 0.0, 0.0, 0.0
            if request.command.HasField("wrench"):
                wrench = request.command.wrench
                force_x = wrench.force.x
                force_y = wrench.force.y
                force_z = wrench.force.z
                # torque_x = wrench.torque.x
                # torque_y = wrench.torque.y
                # torque_z = wrench.torque.z
            elif request.command.HasField("io_command"):
                io_cmd = request.command.io_command
                if io_cmd.HasField("f"):
                    force_x = io_cmd.f.pin1.float_value if io_cmd.f.HasField("pin1") else 0.0
                    force_y = io_cmd.f.pin2.float_value if io_cmd.f.HasField("pin2") else 0.0
                    force_z = io_cmd.f.pin3.float_value if io_cmd.f.HasField("pin3") else 0.0
                    force_x = 0.0 if np.isnan(force_x) else force_x
                    force_y = 0.0 if np.isnan(force_y) else force_y
                    force_z = 0.0 if np.isnan(force_z) else force_z

                elif io_cmd.HasField("e"):
                    if io_cmd.e.HasField("pin1") and io_cmd.e.pin1.int_value == 1 and self.haptic_device is not None:
                        print("Calibrating")
                        self.haptic_device.calibrate()

            if self.haptic_device is not None:
                self.haptic_device.set_force_feedback([force_x, force_y, force_z])

        return return_response
