#! /usr/bin/env python3
import threading
import argparse

from main import create_server, default_device_handler


# @Gooey
def main():
    parser = argparse.ArgumentParser(description="Use other input device as mobileIO")
    parser.add_argument(
        "--device", choices=["touchX", "haptic", "gamepad", "keyboard", "bota"], default="gamepad", help="desired device type"
    )
    parser.add_argument("--family", default="HEBI", help="Family for the input device and imitation modules")
    parser.add_argument("--name", default=None, help="Name for the input device and imitation modules")
    parser.add_argument(
        "--key_config",
        default=None,
        help="path to YAML file that maps IO events to mobileIO channels (only for gamepad)",
    )
    parser.add_argument("--bota_ip", default=None, help="IP address of the force/torque sensor")
    parser.add_argument(
        "--port",
        default=16665,
        help="Optional non-default port number to run on (allowing for multiple devices using the hebi-gateway-client)",
    )
    args = parser.parse_args()

    # Initialize the appropriate controller based on the selected device
    if args.device == "touchX":
        try:
            from touchX_interface import TouchX
        except RuntimeError as e:
            print(f"Error initializing touchX device: {e}")
            print("Please ensure you have the necessary touchX packages and dependencies installed.")
            return
        device_controller = TouchX()
        if args.name is None:
            args.name = "hapticIO"
    elif args.device == "haptic":
        try:
            from hebi_haptic_interface import HebiHaptic
        except RuntimeError as e:
            print(f"Error initializing touchX device: {e}")
            print("Please ensure you have the necessary touchX packages and dependencies installed.")
            return
        device_controller = HebiHaptic()
        if args.name is None:
            args.name = "hebiHaptic"
    elif args.device == "gamepad":
        from gamepad_interface import Gamepad

        if args.key_config is None:
            raise ValueError("Key configuration file must be provided for gamepad control.")
        if not args.key_config.endswith(".yaml"):
            raise ValueError("Key configuration file must be a YAML file.")
        args.key_config = str(args.key_config)
        device_controller = Gamepad(0, args.key_config)
        if args.name is None:
            args.name = "gamepadIO"
    elif args.device == "keyboard":
        raise NotImplementedError("Keyboard control is not implemented yet.")
    elif args.device == "bota":
        from bota_interface import Bota

        if args.bota_ip is None:
            raise ValueError("Bota IP address must be provided for bota control.")
        args.bota_ip = str(args.bota_ip)
        device_controller = Bota(args.bota_ip)
        if args.name is None:
            args.name = "botaIO"
    else:
        raise ValueError(f"Unknown device type: {args.device}")

    if args.port is not None:
        port = int(args.port)
        if port is None or port < 1024 or port > 49151:
            raise ValueError("Port must be valid number between 1024 and 49151.")

    device_thread = threading.Thread(target=device_controller.run)
    group = None

    try:
        group = create_server(
            args.family, [args.name], int(args.port), device_handler=[device_controller.device_handler]
        )

        device_thread.start()

        print("Starting Virtual Module")
        group.run()
    except KeyboardInterrupt:
        print("\nShutting down...")
    finally:
        # Properly shutdown the device controller
        device_controller.shutdown = True

        # Stop the group if it was created
        if group is not None:
            group._enabled = False

        # Wait for device thread to finish
        if device_thread.is_alive():
            device_thread.join(timeout=10.0)
            if device_thread.is_alive():
                print("Warning: Device thread did not shutdown cleanly")
            else:
                print("Device thread shutdown successfully")


if __name__ == "__main__":
    main()
