import socket
from time import time

class UdpConnection:

  HEBI_DEFAULT_TIMEOUT = 5.0
  HEBI_UDP_PORT = 16665

  def __init__(self, hostname, port=None, timeout=None, broadcast=False, broadcast_interface_address=""):
    if port is None:
      port = UdpConnection.HEBI_UDP_PORT
    if timeout is None:
      timeout = UdpConnection.HEBI_DEFAULT_TIMEOUT

    self._hostname = hostname
    self._port = port
    self._sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    self._broadcast = broadcast
    if (broadcast):
      self._sock.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
      self._sock.bind((broadcast_interface_address, port))
    else:
      self._sock.bind(('', port))
    self._sock.settimeout(timeout)
    self._sock.setblocking(1)
    self._timeout = timeout

  def __repr__(self):
    return 'UdpConnection(hostname: {0}, port: {1}, socket: {2})'.format(self._hostname, self._port, self._sock)

  def send(self, buffer, length=None):
    """
    Send data to the address bound to the socket.

    Note: This function returns immediately and does not wait for data to actually be sent.
    """
    if length is None:
      length = len(buffer)
    sl = buffer[0:length]
    dsc = (self._hostname, self._port)
    return self._sock.sendto(sl, 0, dsc)

  def recv(self, buffer, length=None):
    """
    Receive data into the provided buffer.

    Note: This call blocks until data is received, or the timeout has been reached.
    """
    if length is None:
      length = len(buffer)
    try:
      t_start = time()
      t_timeout = t_start + self._timeout

      t_curr = t_start
      while True:
        nbytes, remote_nfo = self._sock.recvfrom_into(buffer, length)
        remote_addr = remote_nfo[0]
        remote_port = remote_nfo[1]

        t_curr = time()
        if t_curr >= t_timeout:
          raise socket.timeout()
        if (self._broadcast or remote_addr == self._hostname):
          return nbytes
    except socket.timeout as to:
      import sys
      sys.stderr.write('UdpConnection.recv_into(buffer={0}, length={1}) timed out after waiting for data from {3} for {2} ms.\n'.format(type(buffer), length, int(self.timeout_seconds * 1000), self._hostname))
      return -1
    except Exception as e:
      import sys
      sys.stderr.write('UdpConnection.recv_into(buffer={0}, length={1}) failed to receive data from {3}.\nException:\n{2}\n'.format(type(buffer), length, e, self._hostname))
      return -1

  @property
  def timeout_seconds(self):
    return self._timeout

  @timeout_seconds.setter
  def timeout_seconds(self, val):
    if val < 0.0:
      raise ValueError('val < 0.0')
    self._timeout = val
    self._sock.settimeout(val)

  @property
  def hostname(self):
    return self._hostname

  @property
  def port(self):
    return self._port
