#!/usr/bin/env python3

# Example to look up and list all modules with a given broadcast
# address

import sys
import math
import ipaddress
from time import sleep, time
from hebi_tools.util import create_broadcast_connection, create_module_connection, new_root_message

argc = len(sys.argv)

if argc != 3 and argc != 2 and argc != 4:
  print('Usage: get_module_list <broadcast ip> [interface] [full debug]')
  print('Example for single interface')
  print('  ./set_poly_spring 10.10.10.255 10.10.10.225')
  print('Example for all interfaces')
  print('  ./set_poly_spring 255.255.255.255')
  print('Example for printing full messages from all interfaces:')
  print('  ./set_poly_spring 255.255.255.255 0.0.0.0 FULL')
  exit(1)

ip = sys.argv[1]
if argc == 2:
  interface = ""
else:
  interface = sys.argv[2]
full_debug = False
if argc == 4 and sys.argv[3] == 'FULL':
  full_debug = True

connection = create_broadcast_connection(ip, interface)

# Build "lookup" message
message_out = new_root_message()
message_in = new_root_message()

message_out.request_firmware_info = True
message_out.request_hardware_info = True
message_out.request_ethernet_info = True
message_out.request_command = True
message_out.request_feedback = True
message_out.request_settings = True
message_out.request_remote_services = True

# Send the message to the module
print('Sending message...')
if not connection.send(message_out):
  print('ERROR: Failed when sending message. Exiting.')
  exit(1)

print("Waiting for responses")
# Try to receive a message. Wait for 2 seconds for all messages
t_end = time() + 2
while (time() < t_end):
  if not connection.recv(message_in):
    print('ERROR: Did not get a response from the module after {0} milliseconds.'.format(connection.timeout_ms))
    exit(1)
  if (message_in.HasField("settings")):
    if (full_debug):
      print(message_in)
    else:
      print("{0} | {1}".format(message_in.settings.name.family, message_in.settings.name.name))
      print("{0}".format(ipaddress.IPv4Address(message_in.ethernet_info.ip_address)))
      print("{0}".format(hex(message_in.ethernet_info.mac_address)))
    print("")