"""
Gamepad controller for virtual mobile IO.
"""

import time
import importlib
import platform
import yaml
import inputs

from hebi_tools.hebi_proto import FirmwareInfo


SYSTEM = platform.system()


class Gamepad:
    AXIS_MAX = pow(2, 15)
    TRIGGER_MAX = 256

    def __init__(self, gamepad_idx, config_file):
        self.gamepad_idx = gamepad_idx
        self.axis_states = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
        self.button_states = [0, 0, 0, 0, 0, 0, 0, 0]
        self.keymap = self.load_config(config_file)
        self.shutdown = False
        try:
            self.gamepad = inputs.devices.gamepads[self.gamepad_idx]
            self.disconnected = False
        except:
            self.gamepad = None
            self.disconnected = True
            print("Waiting for gamepad...")

    def load_config(self, config_file):
        with open(config_file) as cf:
            try:
                return yaml.safe_load(cf)
            except yaml.YAMLError as e:
                print(e)

    def run(self):
        while not self.shutdown:
            if self.disconnected:
                try:
                    importlib.reload(inputs)
                    time.sleep(0.1)
                    self.gamepad = inputs.devices.gamepads[self.gamepad_idx]
                    if self.gamepad is not None:
                        self.disconnected = False
                        print("Connected to gamepad!")
                except KeyboardInterrupt:
                    self.shutdown = True
                    break
                except:
                    continue

            try:
                if SYSTEM == "Windows":
                    self.gamepad._GamePad__check_state()
                events = self.gamepad._do_iter()
            except (OSError, inputs.UnpluggedError):
                print("Lost connection to gamepad...")
                self.disconnected = True
                self.axis_states = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]
                self.button_states = [0, 0, 0, 0, 0, 0, 0, 0]
                continue
            except KeyboardInterrupt:
                self.shutdown = True
                break
            if not events:
                continue
            for event in events:
                if event.code in self.keymap.keys():
                    evt_type = self.keymap[event.code]["type"]
                    idx = self.keymap[event.code]["idx"]
                    if evt_type == "axis":
                        scale = self.keymap[event.code]["scale"]
                        self.axis_states[idx] = float(event.state) / float(scale)
                    elif evt_type == "button":
                        self.button_states[idx] = event.state

    def device_handler(self, device, request, response):
        response.sender_id = device.sender_id
        response.echo.tx_time = request.echo.tx_time
        response.echo.payload = request.echo.payload
        response.echo.sequence_number = request.echo.sequence_number

        return_response = False

        if request.HasField("settings"):
            if request.settings.HasField("name"):
                if request.settings.name.HasField("name"):
                    device.name = request.settings.name.name
                if request.settings.name.HasField("family"):
                    device.family = request.settings.name.family

        if request.request_settings:
            response.settings.name.name = device.name
            response.settings.name.family = device.family
            return_response = True

        else:
            response.settings.ClearField("name")

        if request.request_firmware_info:
            response.firmware_info.type = device.fw_type
            response.firmware_info.revision = device.fw_rev
            response.firmware_info.mode = FirmwareInfo.APPLICATION
            return_response = True
        else:
            response.ClearField("firmware_info")

        if request.request_ethernet_info:
            response.ethernet_info.mac_address = device.mac_address
            response.ethernet_info.ip_address = device.ip_address
            response.ethernet_info.netmask = device.netmask
            return_response = True
        else:
            response.ClearField("ethernet_info")

        if request.request_hardware_info:
            response.hardware_info.serial_number = device.hw_serial_number
            response.hardware_info.mechanical_type = device.hw_mechanical_type
            response.hardware_info.mechanical_revision = device.hw_mechanical_revision
            response.hardware_info.electrical_type = device.hw_electrical_type
            response.hardware_info.electrical_revision = device.hw_electrical_revision
            return_response = True
        else:
            response.ClearField("hardware_info")

        if request.request_feedback and not self.disconnected:
            response.feedback.io_feedback.a.pin1.float_value = self.axis_states[0]
            response.feedback.io_feedback.a.pin2.float_value = self.axis_states[1]
            response.feedback.io_feedback.a.pin3.float_value = self.axis_states[2]
            response.feedback.io_feedback.a.pin4.float_value = self.axis_states[3]
            response.feedback.io_feedback.a.pin5.float_value = self.axis_states[4]
            response.feedback.io_feedback.a.pin6.float_value = self.axis_states[5]
            response.feedback.io_feedback.a.pin7.float_value = self.axis_states[6]
            response.feedback.io_feedback.a.pin8.float_value = self.axis_states[7]

            response.feedback.io_feedback.b.pin1.int_value = self.button_states[0]
            response.feedback.io_feedback.b.pin2.int_value = self.button_states[1]
            response.feedback.io_feedback.b.pin3.int_value = self.button_states[2]
            response.feedback.io_feedback.b.pin4.int_value = self.button_states[3]
            response.feedback.io_feedback.b.pin5.int_value = self.button_states[4]
            response.feedback.io_feedback.b.pin6.int_value = self.button_states[5]
            response.feedback.io_feedback.b.pin7.int_value = self.button_states[6]
            response.feedback.io_feedback.b.pin8.int_value = self.button_states[7]
            return_response = True
        else:
            response.ClearField("feedback")

        return return_response
