import socket
from time import time


class UdpServerConnection:

  HEBI_DEFAULT_TIMEOUT = 5.0
  HEBI_UDP_PORT = 16665

  def __init__(self, port=None, timeout=None):
    if port is None:
      port = UdpServerConnection.HEBI_UDP_PORT
    if timeout is None:
      timeout = UdpServerConnection.HEBI_DEFAULT_TIMEOUT

    self._port = port
    self._sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    self._sock.bind(('', port))
    self._sock.settimeout(timeout)
    self._sock.setblocking(1)
    self._timeout = timeout

  def __repr__(self):
    return 'UdpServerConnection(socket: {0}, port: {1})'.format(self._sock, self._port)

  def send(self, buffer, length=None):
    """
    Send data to the last address received from

    Note: This function returns immediately and does not wait for data to actually be sent.
    """
    if length is None:
      length = len(buffer)
    sl = buffer[0:length]
    dsc = (self._from_hostname, self._from_port)
    return self._sock.sendto(sl, 0, dsc)

  def recv(self, buffer, length=None):
    """
    Receive data into the provided buffer.

    Note: This call blocks until data is received, or the timeout has been reached.
    """
    if length is None:
      length = len(buffer)
    try:
      t_start = time()
      t_timeout = t_start + self._timeout

      t_curr = t_start
      remote_addr = ''
      remote_port = 0 
      while True:
        nbytes, remote_info = self._sock.recvfrom_into(buffer, length)
        remote_addr = remote_info[0]
        remote_port = remote_info[1]

        t_curr = time()
        if t_curr >= t_timeout:
          raise socket.timeout()

        # Save source info for outgoing "send" command
        self._from_hostname = remote_addr
        self._from_port = remote_port
        return nbytes
    except socket.timeout as to:
      import sys
      sys.stderr.write('UdpServerConnection.recv_into(buffer={0}, length={1}) timed out after waiting for data from {3} for {2} ms.\n'.format(type(buffer), length, int(self.timeout_seconds * 1000), remote_addr))
      return -1
    except Exception as e:
      import sys
      sys.stderr.write('UdpServerConnection.recv_into(buffer={0}, length={1}) failed to receive data from {3}.\nException:\n{2}\n'.format(type(buffer), length, e, remote_addr))
      return -1

  @property
  def timeout_seconds(self):
    return self._timeout

  @timeout_seconds.setter
  def timeout_seconds(self, val):
    if val < 0.0:
      raise ValueError('val < 0.0')
    self._timeout = val
    self._sock.settimeout(val)
